using VrpSolver, JuMP, ArgParse
# using CPLEX
include("data.jl")
include("model.jl")

function parse_commandline(args_array::Array{String,1}, appfolder::String)
   s = ArgParseSettings(usage="##### VRPSolver #####\n\n"*
	   "  On interactive mode, call main([\"arg1\", ..., \"argn\"])", exit_after_help=false)
   @add_arg_table s begin
      "instance"
         help = "Instance file path"
      "--cfg", "-c"
         help = "Configuration file path"
         default = "$appfolder/../config/BPP.cfg"
      "--ub","-u"
         help = "Upper bound (primal bound)"
         arg_type = Float64
         default = 1000000.0
      "--batch","-b" 
         help = "batch file path" 
   end
   return parse_args(args_array, s)
end

function run_bpp(app)
   println("Application parameters:")
   for (arg,val) in app
      println("  $arg  =>  $(repr(val))")
   end
   flush(stdout)

   data = readBPPData(app["instance"])
   @show data

   (model, x) = build_model(data)

   # enum_paths, complete_form = get_complete_formulation(model, app["cfg"])
   # complete_form.solver = CplexSolver() # set MIP solver
   # print_enum_paths(enum_paths)
   # println(complete_form)
   # solve(complete_form)
   # println("Objective value: $(getobjectivevalue(complete_form))\n")

   instance_name = split(basename(app["instance"]), ".txt")[1]
   optimizer = VrpOptimizer(model, app["cfg"], instance_name)
   set_cutoff!(optimizer, app["ub"])

   (status, solution_found) = optimize!(optimizer)
   println("########################################################")
   if solution_found # Is there a solution?
      T = items(data) 
      for path_id in 1:get_number_of_positive_paths(optimizer)
         print("bin_id=$(path_id): ")
         for t in T
            if get_value(optimizer, x[t], path_id) > 0.5
               print(data.weights[t], " ")
            end
         end
         println()
      end
      println("Objective value: ", get_objective_value(optimizer))
   elseif status == :Optimal
      println("Problem infeasible")
   else
      println("Solution not found")
   end
   println("########################################################")
end

function main(args)
   appfolder = dirname(@__FILE__)
   app = parse_commandline(args, appfolder)
   isnothing(app) && return
   if app["batch"] != nothing
      for line in readlines(app["batch"])
         if isempty(strip(line)) || strip(line)[1] == '#'
            continue
         end
         args_array = [String(s) for s in split(line)]
         app_line = parse_commandline(args_array, appfolder)
         run_bpp(app_line)
      end
   else
      run_bpp(app)
   end
end

if isempty(ARGS)
   main(["--help"])
else
   main(ARGS)
end
